<?php
declare(strict_types=1);

// Faucet.win installer
$root = dirname(__DIR__);
$configPath = $root . '/config/config.php';

if (is_file($configPath)) {
    echo '<h1>Already installed</h1><p>Delete /install for safety.</p>';
    exit;
}

function esc(string $s): string { return htmlspecialchars($s, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); }

$err = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db_host = trim((string)($_POST['db_host'] ?? 'localhost'));
    $db_name = trim((string)($_POST['db_name'] ?? ''));
    $db_user = trim((string)($_POST['db_user'] ?? ''));
    $db_pass = (string)($_POST['db_pass'] ?? '');

    $site_name = trim((string)($_POST['site_name'] ?? 'faucet.win'));
    $site_base_url = trim((string)($_POST['site_base_url'] ?? ''));

    $turnstile_sitekey = trim((string)($_POST['turnstile_sitekey'] ?? ''));
    $turnstile_secret = trim((string)($_POST['turnstile_secret'] ?? ''));

    $mail_from = trim((string)($_POST['mail_from'] ?? ''));

    $admin_email = mb_strtolower(trim((string)($_POST['admin_email'] ?? '')));
    $admin_pass  = (string)($_POST['admin_pass'] ?? '');

    if ($db_name==='' || $db_user==='' || !filter_var($admin_email, FILTER_VALIDATE_EMAIL) || strlen($admin_pass) < 12) {
        $err = 'Please fill all required fields. Admin password must be 12+ chars.';
    } else {
        try {
            $dsn = sprintf('mysql:host=%s;dbname=%s;charset=utf8mb4', $db_host, $db_name);
            $pdo = new PDO($dsn, $db_user, $db_pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            ]);

            $pdo->exec("SET sql_mode='STRICT_ALL_TABLES'");

            // Create tables
            $sql = file_get_contents(__DIR__ . '/schema.sql');
            if ($sql === false) throw new RuntimeException('Missing schema.sql');
            $pdo->exec($sql);

            $app_key = bin2hex(random_bytes(32));

            // Write config
            if (!is_dir($root . '/config')) { mkdir($root . '/config', 0755, true); }
            $cfg = <<<PHP2
<?php
return [
  'db_host' => {DB_HOST},
  'db_name' => {DB_NAME},
  'db_user' => {DB_USER},
  'db_pass' => {DB_PASS},
  'app_key' => {APP_KEY},
  'turnstile_sitekey' => {TS_SITE},
  'turnstile_secret' => {TS_SECRET},
  'mail_from' => {MAIL_FROM},
];
PHP2;
            $repl = function($v) { return var_export($v, true); };
            $cfg = str_replace(
                ['{DB_HOST}','{DB_NAME}','{DB_USER}','{DB_PASS}','{APP_KEY}','{TS_SITE}','{TS_SECRET}','{MAIL_FROM}'],
                [$repl($db_host),$repl($db_name),$repl($db_user),$repl($db_pass),$repl($app_key),$repl($turnstile_sitekey),$repl($turnstile_secret),$repl($mail_from)],
                $cfg
            );
            file_put_contents($configPath, $cfg, LOCK_EX);

            // Seed settings
            $stmt = $pdo->prepare('INSERT INTO settings(`key`,`value`) VALUES(:k,:v) ON DUPLICATE KEY UPDATE value=VALUES(value)');
            $stmt->execute([':k'=>'site_name', ':v'=>$site_name]);
            $stmt->execute([':k'=>'site_base_url', ':v'=>$site_base_url]);
            $stmt->execute([':k'=>'claim_interval_minutes', ':v'=>'15']);
            $stmt->execute([':k'=>'max_claims_per_day', ':v'=>'5']);
            $stmt->execute([':k'=>'daily_user_cap_doge', ':v'=>'0']);
            $stmt->execute([':k'=>'daily_site_cap_doge', ':v'=>'0']);
            $payout = json_encode([
                ['min'=>1,'max'=>5,'reward'=>'0.50'],
                ['min'=>6,'max'=>20,'reward'=>'0.15'],
                ['min'=>21,'max'=>60,'reward'=>'0.05'],
                ['min'=>61,'max'=>100,'reward'=>'0.01'],
            ], JSON_UNESCAPED_SLASHES);
            $stmt->execute([':k'=>'payout_table_json', ':v'=>$payout]);
            $stmt->execute([':k'=>'maintenance_mode', ':v'=>'0']);
            $stmt->execute([':k'=>'maintenance_message', ':v'=>'']);

            // Create admin
            $hash = password_hash($admin_pass, PASSWORD_ARGON2ID);
            $pdo->prepare('INSERT INTO users(email,password_hash,role,status,email_verified_at,created_at,updated_at,balance_doge,next_claim_at) VALUES(:e,:p,\'admin\',\'active\',NOW(),NOW(),NOW(),0,NOW())')
                ->execute([':e'=>$admin_email, ':p'=>$hash]);

            echo '<h1>Installed!</h1>';
            echo '<p><a href="/">Go to site</a> | <a href="/admin">Admin panel</a></p>';
            echo '<p><b>IMPORTANT:</b> Delete the <code>/install</code> directory now.</p>';
            exit;
        } catch (Throwable $e) {
            $err = 'Install failed: ' . esc($e->getMessage());
        }
    }
}
?>
<!doctype html>
<html><head><meta charset="utf-8"><title>Install - Faucet.win</title></head>
<body style="font-family:system-ui;max-width:980px;margin:40px auto;">
<h1>Faucet.win Installer</h1>
<?php if ($err): ?><p style="color:#b00;"><b><?= $err ?></b></p><?php endif; ?>
<form method="post" style="display:grid;gap:14px;">
  <fieldset style="padding:14px;"><legend><b>Database (MySQL)</b></legend>
    <div style="display:grid;grid-template-columns:repeat(2,1fr);gap:12px;">
      <label>DB Host<br><input name="db_host" value="<?= esc($_POST['db_host'] ?? 'localhost') ?>" required style="width:100%;padding:10px;"></label>
      <label>DB Name<br><input name="db_name" value="<?= esc($_POST['db_name'] ?? '') ?>" required style="width:100%;padding:10px;"></label>
      <label>DB User<br><input name="db_user" value="<?= esc($_POST['db_user'] ?? '') ?>" required style="width:100%;padding:10px;"></label>
      <label>DB Pass<br><input name="db_pass" value="<?= esc($_POST['db_pass'] ?? '') ?>" type="password" style="width:100%;padding:10px;"></label>
    </div>
  </fieldset>

  <fieldset style="padding:14px;"><legend><b>Site</b></legend>
    <div style="display:grid;grid-template-columns:repeat(2,1fr);gap:12px;">
      <label>Site Name<br><input name="site_name" value="<?= esc($_POST['site_name'] ?? 'faucet.win') ?>" required style="width:100%;padding:10px;"></label>
      <label>Site Base URL (for email verification links)<br><input name="site_base_url" placeholder="https://faucet.win" value="<?= esc($_POST['site_base_url'] ?? '') ?>" style="width:100%;padding:10px;"></label>
    </div>
  </fieldset>

  <fieldset style="padding:14px;"><legend><b>Bot Protection (Cloudflare Turnstile)</b></legend>
    <div style="display:grid;grid-template-columns:repeat(2,1fr);gap:12px;">
      <label>Turnstile Site Key<br><input name="turnstile_sitekey" value="<?= esc($_POST['turnstile_sitekey'] ?? '') ?>" style="width:100%;padding:10px;"></label>
      <label>Turnstile Secret Key<br><input name="turnstile_secret" value="<?= esc($_POST['turnstile_secret'] ?? '') ?>" style="width:100%;padding:10px;"></label>
    </div>
    <p style="color:#666;">Leave blank to disable (not recommended).</p>
  </fieldset>

  <fieldset style="padding:14px;"><legend><b>Email</b></legend>
    <label>From Address (optional)<br><input name="mail_from" value="<?= esc($_POST['mail_from'] ?? '') ?>" placeholder="no-reply@faucet.win" style="width:100%;padding:10px;"></label>
  </fieldset>

  <fieldset style="padding:14px;"><legend><b>Admin Account</b></legend>
    <div style="display:grid;grid-template-columns:repeat(2,1fr);gap:12px;">
      <label>Admin Email<br><input name="admin_email" value="<?= esc($_POST['admin_email'] ?? '') ?>" required style="width:100%;padding:10px;"></label>
      <label>Admin Password (12+ chars)<br><input name="admin_pass" type="password" required style="width:100%;padding:10px;"></label>
    </div>
  </fieldset>

  <button type="submit" style="padding:14px;font-size:16px;">Install</button>
</form>
<p style="margin-top:18px;color:#666;">After install: delete <code>/install</code>.</p>
</body></html>
