<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\User;
use App\Services\Auth;
use App\Services\CSRF;
use App\Services\DB;
use App\Services\Mailer;
use App\Services\RateLimiter;
use App\Services\Turnstile;
use App\Services\Util;
use App\Services\View;

final class AuthController
{
    public function showRegister(): void
    {
        View::render('user/register', ['csrf' => CSRF::token()]);
    }

    public function register(): void
    {
        CSRF::validateOrFail();

        $ip = Util::clientIp();
        $bucket = 'reg:' . Util::hmac($ip);
        if (!RateLimiter::hit($bucket, (int)($_ENV['REG_PER_HOUR'] ?? 8), 3600)) {
            http_response_code(429);
            echo 'Too many registrations from this IP. Try later.';
            return;
        }

        $email = trim((string)($_POST['email'] ?? ''));
        $pass  = (string)($_POST['password'] ?? '');
        $tsToken = (string)($_POST['cf_turnstile_response'] ?? '');

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            echo 'Invalid email';
            return;
        }
        $email = mb_strtolower($email);

        // Basic password policy
        if (strlen($pass) < 10) {
            echo 'Password must be at least 10 characters.';
            return;
        }

        // Turnstile (if configured)
        if (!Turnstile::verify($tsToken, $ip)) {
            echo 'Bot check failed';
            return;
        }

        if (User::findByEmail($email)) {
            echo 'Account already exists for this email.';
            return;
        }

        $hash = password_hash($pass, PASSWORD_ARGON2ID);
        $userId = User::create($email, $hash);

        $this->sendVerification($userId, $email);

        echo 'Registered! Check your email to verify your account.';
    }

    public function verify(): void
    {
        $token = (string)($_GET['token'] ?? '');
        if ($token === '') { echo 'Missing token'; return; }

        $pdo = DB::pdo();
        $th = hash('sha256', $token);
        $stmt = $pdo->prepare('SELECT id,user_id,expires_at,used_at FROM email_verifications WHERE token_hash=:t LIMIT 1');
        $stmt->execute([':t'=>$th]);
        $row = $stmt->fetch();
        if (!$row || $row['used_at']) {
            echo 'Invalid or used token.';
            return;
        }
        if (strtotime($row['expires_at']) < time()) {
            echo 'Token expired.';
            return;
        }

        $pdo->beginTransaction();
        $upd = $pdo->prepare('UPDATE email_verifications SET used_at=NOW() WHERE id=:id');
        $upd->execute([':id'=>$row['id']]);
        User::markVerified((int)$row['user_id']);
        $pdo->commit();

        echo 'Email verified. You can now log in.';
    }

    public function resendVerification(): void
    {
        CSRF::validateOrFail();
        $ip = Util::clientIp();
        $bucket = 'resend:' . Util::hmac($ip);
        if (!RateLimiter::hit($bucket, 5, 3600)) {
            http_response_code(429);
            echo 'Too many requests. Try later.';
            return;
        }

        $email = mb_strtolower(trim((string)($_POST['email'] ?? '')));
        $u = User::findByEmail($email);
        if (!$u) { echo 'If that account exists, we sent a link.'; return; }
        if (!empty($u['email_verified_at'])) { echo 'Already verified.'; return; }
        $this->sendVerification((int)$u['id'], $email);
        echo 'Verification email sent.';
    }

    private function sendVerification(int $userId, string $email): void
    {
        $token = bin2hex(random_bytes(24));
        $tokenHash = hash('sha256', $token);
        $pdo = DB::pdo();
        $stmt = $pdo->prepare('INSERT INTO email_verifications(user_id, token_hash, expires_at, created_at) VALUES(:u,:t,DATE_ADD(NOW(), INTERVAL 30 MINUTE),NOW())');
        $stmt->execute([':u'=>$userId, ':t'=>$tokenHash]);

        $base = rtrim((string)\App\Models\Setting::get('site_base_url', ''), '/');
        if ($base === '') {
            $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
            $base = $scheme . '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost');
        }
        $link = $base . '/verify?token=' . urlencode($token);
        $subj = 'Verify your Faucet.win account';
        $html = '<p>Click to verify your email:</p><p><a href="' . Util::esc($link) . '">' . Util::esc($link) . '</a></p><p>This link expires in 30 minutes.</p>';
        Mailer::send($email, $subj, $html);
    }

    public function showLogin(): void
    {
        View::render('user/login', ['csrf' => CSRF::token()]);
    }

    public function login(): void
    {
        CSRF::validateOrFail();

        $ip = Util::clientIp();
        $bucket = 'login:' . Util::hmac($ip);
        if (!RateLimiter::hit($bucket, 25, 3600)) {
            http_response_code(429);
            echo 'Too many login attempts. Try later.';
            return;
        }

        $email = mb_strtolower(trim((string)($_POST['email'] ?? '')));
        $pass  = (string)($_POST['password'] ?? '');

        $u = User::findByEmail($email);
        if (!$u || !password_verify($pass, (string)$u['password_hash'])) {
            echo 'Invalid credentials';
            return;
        }
        if (($u['status'] ?? 'active') !== 'active') {
            echo 'Account not active';
            return;
        }

        Auth::login((int)$u['id']);
        User::updateLoginMeta((int)$u['id'], Util::hmac($ip), Util::hmac(Util::userAgent()));
        Util::redirect('/dashboard');
    }

    public function logout(): void
    {
        CSRF::validateOrFail();
        Auth::logout();
        Util::redirect('/');
    }
}
