<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\Setting;
use App\Services\Auth;
use App\Services\CSRF;
use App\Services\DB;
use App\Services\Util;
use App\Services\View;

final class AdminController
{
    public function showLogin(): void
    {
        View::render('admin/login', ['csrf'=>CSRF::token()]);
    }

    public function login(): void
    {
        CSRF::validateOrFail();
        $email = mb_strtolower(trim((string)($_POST['email'] ?? '')));
        $pass = (string)($_POST['password'] ?? '');
        $pdo = DB::pdo();
        $stmt = $pdo->prepare('SELECT * FROM users WHERE email=:e LIMIT 1');
        $stmt->execute([':e'=>$email]);
        $u = $stmt->fetch();
        if (!$u || ($u['role'] ?? '') !== 'admin' || !password_verify($pass, (string)$u['password_hash'])) {
            echo 'Invalid admin credentials';
            return;
        }
        \App\Services\Auth::login((int)$u['id']);
        Util::redirect('/admin');
    }

    public function logout(): void
    {
        CSRF::validateOrFail();
        \App\Services\Auth::logout();
        Util::redirect('/');
    }

    public function dashboard(): void
    {
        $admin = Auth::requireAdmin();
        $pdo = DB::pdo();
        $today = date('Y-m-d');
        $claimsToday = $pdo->prepare('SELECT COUNT(*) c, COALESCE(SUM(reward_doge),0) s FROM claims WHERE claim_date=:d');
        $claimsToday->execute([':d'=>$today]);
        $row = $claimsToday->fetch();
        $users = $pdo->query('SELECT COUNT(*) c FROM users')->fetch();

        View::render('admin/dashboard', [
            'admin'=>$admin,
            'csrf'=>CSRF::token(),
            'claimsCount'=>(int)($row['c']??0),
            'claimsSum'=>(string)($row['s']??'0'),
            'usersCount'=>(int)($users['c']??0),
        ]);
    }

    public function settings(): void
    {
        $admin = Auth::requireAdmin();
        View::render('admin/settings', [
            'admin'=>$admin,
            'csrf'=>CSRF::token(),
            'settings'=>Setting::all(),
        ]);
    }

    public function saveSettings(): void
    {
        $admin = Auth::requireAdmin();
        CSRF::validateOrFail();

        $keys = [
            'site_name','site_base_url','claim_interval_minutes','max_claims_per_day',
            'daily_user_cap_doge','daily_site_cap_doge','payout_table_json','maintenance_mode','maintenance_message'
        ];

        foreach ($keys as $k) {
            if (!array_key_exists($k, $_POST)) continue;
            $v = $_POST[$k];
            if ($k === 'payout_table_json') {
                $decoded = json_decode((string)$v, true);
                if (!is_array($decoded)) {
                    echo 'Invalid payout table JSON';
                    return;
                }
                Setting::set($k, $decoded);
            } else {
                Setting::set($k, is_string($v) ? trim($v) : $v);
            }
        }

        // Audit
        $pdo = DB::pdo();
        $pdo->prepare('INSERT INTO admin_audit(admin_user_id, action, meta_json, ip_hash, created_at) VALUES(:a,:act,:m,:ip,NOW())')
            ->execute([
                ':a'=>(int)$admin['id'],
                ':act'=>'settings_update',
                ':m'=>json_encode(['keys'=>$keys], JSON_UNESCAPED_SLASHES),
                ':ip'=>Util::hmac(Util::clientIp()),
            ]);

        Util::redirect('/admin/settings');
    }

    public function users(): void
    {
        $admin = Auth::requireAdmin();
        $pdo = DB::pdo();
        $q = trim((string)($_GET['q'] ?? ''));
        if ($q !== '') {
            $stmt = $pdo->prepare('SELECT id,email,role,status,email_verified_at,balance_doge,next_claim_at,created_at FROM users WHERE email LIKE :q ORDER BY id DESC LIMIT 200');
            $stmt->execute([':q'=>'%' . $q . '%']);
            $rows = $stmt->fetchAll();
        } else {
            $rows = $pdo->query('SELECT id,email,role,status,email_verified_at,balance_doge,next_claim_at,created_at FROM users ORDER BY id DESC LIMIT 200')->fetchAll();
        }
        View::render('admin/users', [
            'admin'=>$admin,
            'csrf'=>CSRF::token(),
            'users'=>$rows,
            'q'=>$q,
        ]);
    }
}
